% run_n1_opf.m
% 主脚本：执行N-1安全约束的最优潮流分析

% 清理环境
clear;
clc;

% 添加MATPOWER路径并定义常量
addpath('D:\program\matlabR2023a\bin\matpower8.0');  % 替换为MATPOWER所在的路径
define_constants;

% 加载case9电网数据
mpc = loadcase('case9');  % 确保你已经下载并保存了case9文件

% 获取有功和无功功率（需求）
p_default = mpc.bus(:, PD);  % 有功功率（MW）
q_default = mpc.bus(:, QD);  % 无功功率（MVAr）

% 获取bus数据
bus = mpc.bus;

% 获取支路数据
branches = mpc.branch;

% 获取发电机数据
gens = mpc.gen;

% 获取baseMVA
baseMVA = mpc.baseMVA;

% 获取电阻和电抗
R = branches(:, 3);  % 电阻
X = branches(:, 4);  % 电抗

% 获取发电机数量
num_gens = size(gens, 1);

% 获取支路数量
num_branches = size(branches, 1);

% 定义故障类型
fault_types = {'Line', 'Generator'};  % 'Line' 表示传输线路故障，'Generator' 表示发电机故障

% 设置需要满足的个数
expect_num = 100;

% 初始化结构体来保存结果
% 预分配空间以提高效率
max_faults = num_branches + num_gens;
% results_data = repmat(struct('type', [], 'id', [], 'success', [], 'voltage_magnitude', [], ...
%     'voltage_angle', [], 'p_sampled', [], 'q_sampled', [], 'Pg', [], 'Qg', [], ...
%     'G_fault', [], 'B_fault', [], 'adj_matrix', [], 'node_features', [], 'edge_features_norm', []), max_faults, 1);
successful_tot_num = 0;  % 全局成功求解次数
branch_ids = [];  % 用于记录未被孤岛效应跳过的故障支路ID
gen_ids = [];     % 用于记录未被孤岛效应跳过的故障发电机ID

% 使用常规for循环遍历所有故障场景
for fault_idx = 1:(num_branches + num_gens )
    % 初始化临时结果
    temp_result = struct();
    
    % 根据fault_idx确定故障类型和ID
    if fault_idx <= num_branches
        fault_type = 'Line';
        fault_id = fault_idx;
    else
        fault_type = 'Generator';
        fault_id = fault_idx - num_branches;
    end
    
    % 确定总故障数
    if strcmp(fault_type, 'Line')
        total_faults = num_branches;
    elseif strcmp(fault_type, 'Generator')
        total_faults = num_gens;
    else
        total_faults = 1;  % 默认值，避免潜在错误
    end
    
    % 打印当前故障信息
    fprintf('模拟故障类型: %s, 故障ID: %d/%d\n', fault_type, fault_id, total_faults);
    
    % 创建故障后的电网数据
    if strcmp(fault_type, 'Line')
        mpc_fault = simulate_line_fault(mpc, fault_id);
    elseif strcmp(fault_type, 'Generator')
        mpc_fault = simulate_gen_fault(mpc, fault_id);
    end
    
    % 判断故障是否导致孤岛效应
    if check_islanding(mpc_fault)
        fprintf(['故障类型: ', fault_type, ', ID: ', num2str(fault_id), ...
            ' 导致孤岛效应。跳过该故障场景。\n']);
        continue;  % 跳过当前回合，进行下一次故障模拟
    end
    
    % 记录通过孤岛效应检查的故障ID
    if strcmp(fault_type, 'Line')
        branch_ids = [branch_ids; fault_id];
        fault_id_store = fault_id;
    elseif strcmp(fault_type, 'Generator')
        gen_ids = [gen_ids; 0];
        fault_id_store = 0;
    end
    
    % 进行功率采样和OPF求解
    successful_runs = 0;  % 当前故障场景的成功次数
    
    while successful_runs < expect_num
        % 随机生成有功和无功功率采样系数
        p_sampled = p_default .* (0.8 + 0.4 * rand(size(p_default)));  % 采样后有功功率
        q_sampled = q_default .* (0.8 + 0.4 * rand(size(q_default)));  % 采样后无功功率

        % 替换mpc_fault中的有功和无功功率
        mpc_fault.bus(:, PD) = p_sampled;  % 替换有功功率
        mpc_fault.bus(:, QD) = q_sampled;  % 替换无功功率

        % 运行OPF
        [result, G_fault, B_fault] = perform_opf(mpc_fault, fault_type, fault_id, R, X);

        % 检查求解是否成功
        if result.success
            % 增加全局成功求解次数
            successful_tot_num = successful_tot_num + 1;

            % 保存当前求解结果
            results_data(successful_tot_num).type = fault_type;
            results_data(successful_tot_num).id = fault_id_store;
            results_data(successful_tot_num).success = true;
            results_data(successful_tot_num).voltage_magnitude = result.bus(:, VM);  % 电压幅值 (p.u.)
            results_data(successful_tot_num).voltage_angle = result.bus(:, VA);      % 电压相角 (度)
            results_data(successful_tot_num).p_sampled = p_sampled;                 % 采样后的有功功率
            results_data(successful_tot_num).q_sampled = q_sampled;                 % 采样后的无功功率
            results_data(successful_tot_num).Pg = result.gen(:, PG);                % 发电机有功功率 (MW)
            results_data(successful_tot_num).Qg = result.gen(:, QG);                % 发电机无功功率 (MVAr)
            results_data(successful_tot_num).G_fault = G_fault;                     % 故障情况下的电导
            results_data(successful_tot_num).B_fault = B_fault;                     % 故障情况下的电纳


            % 生成图数据
            
            
            [adj_matrix, node_features, edge_features_norm] = generate_graph(mpc_fault, p_sampled, q_sampled, gens);
            
            % 将图数据保存到 results_data 结构体中
            results_data(successful_tot_num).adj_matrix = adj_matrix;
            results_data(successful_tot_num).node_features = node_features;
            results_data(successful_tot_num).edge_features_norm = edge_features_norm;

            % 增加成功计数
            successful_runs = successful_runs + 1;

            fprintf('成功求解: %d/%d\n', successful_runs, expect_num);
        else
            % 如果求解失败，尝试下一次采样
            fprintf('求解失败，重新采样...\n');
        end
    end
end

% 保存结果到.mat文件
save('N1_OPF_results.mat', 'results_data', 'successful_tot_num', 'branch_ids', 'gen_ids', 'baseMVA', 'bus', 'gens');

disp('N-1 OPF analysis complete. Results saved to "N1_OPF_results.mat".');
